//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3
import QtQuick.Controls 1.0
import QtGraphicalEffects 1.0

import wesual.Controls 1.0

Rectangle {
    id : toolbutton

    property Action action
    property int    iconSize          : 24
    property alias  showLabel         : label.visible
    property bool   centeredStyle     : false
    property bool   down              : false

    activeFocusOnTab : true

    QtObject {
        id : p_

        property color baseColor : toolbutton.color
        readonly property int   animationDuration : 0
    }

    width : {
        if (showLabel) {
            if (!centeredStyle)
                return (icon.width + Math.ceil(label.implicitWidth) + 17);
            else
                return (Math.ceil(label.implicitWidth) + 20);
        } else
            return (icon.width + 4)
    }
    height : centeredStyle ? 50 : (iconSize + 6)
    color   : UI.color(UI.PrimaryBase)
    enabled : action && action.enabled

    Behavior on enabled {
        PropertyAnimation { duration : 0 }
    }

    Image {
        id : icon

        anchors.verticalCenter : {
            if (!centeredStyle)
                return parent ? parent.verticalCenter : undefined;
        }
        anchors.horizontalCenter : {
            if (!showLabel && centeredStyle) {
                return parent ? parent.horizontalCenter : undefined;
            }
        }


        y : centeredStyle  ? 4 : 0
        x : !centeredStyle ? 2 : 0
        width  : iconSize
        height : iconSize
        sourceSize {
            width  : iconSize
            height : iconSize
        }
        source : {
            if (!action)
                return "";

            if (action.iconName) {
                return "qrc:/controls/icons/" + action.iconName + ".png"
            } else {
                return action.iconSource;
            }
        }
        visible : parent.enabled || toolbutton.down
        mipmap : true
    }

    HueSaturation {
        anchors.fill : icon
        source       : icon
        saturation   : -1.0
        lightness    : 0.79
        visible      : !parent.enabled && !toolbutton.down
    }

    Text {
        id : label

        anchors.verticalCenter : {
            if (!centeredStyle)
                return parent.verticalCenter;
        }
        anchors.horizontalCenter : {
            if (centeredStyle)
                return parent.horizontalCenter;
        }
        font {
            family : UI.fontFamily(UI.PrimaryFont)
            weight : UI.fontWeight(UI.PrimaryFont)
            pixelSize : 14
        }
        width : Math.min(parent.width - x, implicitWidth)
        x : centeredStyle ? 0 : (iconSize + 7)
        y : centeredStyle ? (41 - label.font.pixelSize) : 0
        text : action ? action.text : ""
        color : {
            if (!parent.enabled && !toolbutton.down)
                return UI.color(UI.DisabledText);
            else if (mouseArea.pressed || toolbutton.down)
                return UI.color(UI.SelectedTextColor);
            else
                return UI.color(UI.PrimaryTextColor);
        }
        elide : Text.ElideRight

        Behavior on color {
            ColorAnimation { duration : 0 }
        }
    }

    states : [
        State {
            name : "clicked"
            when : mouseArea.pressed || down

            PropertyChanges {
                target : toolbutton
                color : UI.color(UI.PrimaryPress)
            }
        },
        State {
            name : "hovered"
            when : (ha.hovered && !mouseArea.pressed && !down) || toolbutton.activeFocus
            PropertyChanges {
                target : toolbutton
                color : UI.color(UI.PrimaryHover)
            }
        }
    ]

    transitions : [
        Transition {
            ColorAnimation {
                duration : p_.animationDuration
            }
        }
    ]

    MouseArea {
        // catch rightclick of toolbutton area without press/click highlight
        anchors.fill : parent
        acceptedButtons : Qt.RightButton
        onClicked : {}
    }

    MouseArea {
        id : mouseArea

        anchors.fill : parent
        onClicked : {
            if (action)
                action.trigger();
        }
        cursorShape : Qt.PointingHandCursor
    }

    HoverArea {
        id : ha
        anchors.fill : parent
    }

    // Key handling
    Keys.onEnterPressed : {
        if (action)
            action.trigger();
    }
    Keys.onReturnPressed : {
        if (action)
            action.trigger();
    }
}
